/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.http.response;

import static org.apache.juneau.http.response.PermanentRedirect.*;

import java.net.*;
import java.util.*;

import org.apache.http.*;
import org.apache.http.Header;
import org.apache.juneau.annotation.*;
import org.apache.juneau.http.*;
import org.apache.juneau.http.annotation.*;
import org.apache.juneau.http.header.*;

/**
 * Represents an <c>HTTP 308 Permanent Redirect</c> response.
 *
 * <p>
 * The request and all future requests should be repeated using another URI. 307 and 308 parallel the behaviors of 302 and 301, but do not allow the HTTP method to change.
 * So, for example, submitting a form to a permanently redirected resource may continue smoothly.
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/JuneauRestCommonBasics">juneau-rest-common Basics</a>
 * </ul>
 */
@Response
@StatusCode(STATUS_CODE)
@Schema(description = REASON_PHRASE)
public class PermanentRedirect extends BasicHttpResponse {

	/** HTTP status code */
	public static final int STATUS_CODE = 308;

	/** Reason phrase */
	public static final String REASON_PHRASE = "Permanent Redirect";

	/** Default status line */
	private static final BasicStatusLine STATUS_LINE = BasicStatusLine.create(STATUS_CODE, REASON_PHRASE);

	/** Default unmodifiable instance */
	public static final PermanentRedirect INSTANCE = new PermanentRedirect().setUnmodifiable();

	/**
	 * Constructor.
	 */
	public PermanentRedirect() {
		super(STATUS_LINE);
	}

	/**
	 * Constructor.
	 *
	 * <p>
	 * This is the constructor used when parsing an HTTP response.
	 *
	 * @param response The HTTP response to copy from.  Must not be <jk>null</jk>.
	 * @throws AssertionError If HTTP response status code does not match what was expected.
	 */
	public PermanentRedirect(HttpResponse response) {
		super(response);
		assertStatusCode(response);
	}

	/**
	 * Copy constructor.
	 *
	 * @param copyFrom The bean to copy from.
	 */
	public PermanentRedirect(PermanentRedirect copyFrom) {
		super(copyFrom);
	}

	/**
	 * Creates a builder for this class initialized with the contents of this bean.
	 *
	 * @return A new builder bean.
	 */
	public PermanentRedirect copy() {
		return new PermanentRedirect(this);
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setContent(HttpEntity value) {
		super.setContent(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setContent(String value) {
		super.setContent(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setHeader2(Header value) {
		super.setHeader2(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setHeader2(String name, String value) {
		super.setHeader2(name, value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setHeaders(HeaderList value) {
		super.setHeaders(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setHeaders(List<Header> values) {
		super.setHeaders(values);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setHeaders2(Header...values) {
		super.setHeaders2(values);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setLocale2(Locale value) {
		super.setLocale2(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setLocation(String value) {
		super.setLocation(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setLocation(URI value) {
		super.setLocation(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setProtocolVersion(ProtocolVersion value) {
		super.setProtocolVersion(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setReasonPhrase2(String value) {
		super.setReasonPhrase2(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setReasonPhraseCatalog(ReasonPhraseCatalog value) {
		super.setReasonPhraseCatalog(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setStatusCode2(int value) {
		super.setStatusCode2(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setStatusLine(BasicStatusLine value) {
		super.setStatusLine(value);
		return this;
	}

	@Override /* Overridden from BasicHttpResponse */
	public PermanentRedirect setUnmodifiable() {
		super.setUnmodifiable();
		return this;
	}
}