/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.sql.engine.rel;

import static org.apache.ignite.internal.sql.engine.trait.TraitUtils.changeTraits;

import java.util.List;
import org.apache.calcite.plan.RelOptCluster;
import org.apache.calcite.plan.RelOptTable;
import org.apache.calcite.plan.RelTraitSet;
import org.apache.calcite.rel.RelCollation;
import org.apache.calcite.rel.RelInput;
import org.apache.calcite.rel.RelWriter;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.util.ImmutableIntList;
import org.apache.ignite.internal.sql.engine.prepare.bounds.SearchBounds;
import org.apache.ignite.internal.sql.engine.rel.explain.IgniteRelWriter;
import org.apache.ignite.internal.sql.engine.schema.IgniteIndex;
import org.apache.ignite.internal.sql.engine.schema.IgniteIndex.Type;
import org.jetbrains.annotations.Nullable;

/**
 * Relational operator that returns the contents of a table using an index.
 */
public class IgniteIndexScan extends AbstractIndexScan implements SourceAwareIgniteRel {
    private static final String REL_TYPE_NAME = "IndexScan";

    private final long sourceId;

    /** Index collation. Required for proper expanding search bounds and creating index row converter. */
    private final RelCollation collation;

    /**
     * Constructor used for deserialization.
     *
     * @param input Serialized representation.
     */
    public IgniteIndexScan(RelInput input) {
        super(changeTraits(input, IgniteConvention.INSTANCE));

        collation = input.getCollation();

        Object srcIdObj = input.get("sourceId");
        if (srcIdObj != null) {
            sourceId = ((Number) srcIdObj).longValue();
        } else {
            sourceId = -1;
        }
    }

    /**
     * Creates a IndexScan.
     *
     * @param cluster Cluster that this relational expression belongs to
     * @param traits Traits of this relational expression
     * @param tbl Table definition.
     * @param idxName Index name.
     * @param type Type of the index.
     * @param collation Index columns collation.
     * @param names Names of the fields in output row. If not provided, field names will be autogenerated.
     * @param proj Projects.
     * @param cond Filters.
     * @param searchBounds Index search conditions.
     * @param requiredCols Participating columns.
     */
    public IgniteIndexScan(
            RelOptCluster cluster,
            RelTraitSet traits,
            RelOptTable tbl,
            String idxName,
            IgniteIndex.Type type,
            RelCollation collation,
            @Nullable List<String> names,
            @Nullable List<RexNode> proj,
            @Nullable RexNode cond,
            @Nullable List<SearchBounds> searchBounds,
            @Nullable ImmutableIntList requiredCols
    ) {
        this(-1L, cluster, traits, tbl, idxName, type, collation, names, proj, cond, searchBounds, requiredCols);
    }

    /**
     * Creates a IndexScan.
     *
     * @param sourceId Sorce id.
     * @param cluster Cluster that this relational expression belongs to
     * @param traits Traits of this relational expression
     * @param tbl Table definition.
     * @param idxName Index name.
     * @param type Type of the index.
     * @param collation Index columns collation.
     * @param names Names of the fields in output row. If not provided, field names will be autogenerated.
     * @param proj Projects.
     * @param cond Filters.
     * @param searchBounds Index search conditions.
     * @param requiredCols Participating columns.
     */
    private IgniteIndexScan(
            long sourceId,
            RelOptCluster cluster,
            RelTraitSet traits,
            RelOptTable tbl,
            String idxName,
            IgniteIndex.Type type,
            RelCollation collation,
            @Nullable List<String> names,
            @Nullable List<RexNode> proj,
            @Nullable RexNode cond,
            @Nullable List<SearchBounds> searchBounds,
            @Nullable ImmutableIntList requiredCols
    ) {
        super(cluster, traits, List.of(), tbl, idxName, type, names, proj, cond, searchBounds, requiredCols);

        this.sourceId = sourceId;
        this.collation = collation;
    }

    /** {@inheritDoc} */
    @Override
    public long sourceId() {
        return sourceId;
    }

    /** {@inheritDoc} */
    @Override
    public RelCollation collation() {
        return collation;
    }

    /** {@inheritDoc} */
    @Override
    protected RelWriter explainTerms0(RelWriter pw) {
        return super.explainTerms0(pw)
                .itemIf("sourceId", sourceId, sourceId != -1)
                .itemIf("collation", collation(), type == Type.SORTED);
    }

    /** {@inheritDoc} */
    @Override
    public <T> T accept(IgniteRelVisitor<T> visitor) {
        return visitor.visit(this);
    }

    /** {@inheritDoc} */
    @Override
    public IgniteRel clone(long sourceId) {
        return new IgniteIndexScan(sourceId, getCluster(), getTraitSet(), getTable(),
                idxName, type, collation, names, projects, condition, searchBounds, requiredColumns);
    }

    /** {@inheritDoc} */
    @Override
    public IgniteRel clone(RelOptCluster cluster, List<IgniteRel> inputs) {
        return new IgniteIndexScan(sourceId, cluster, getTraitSet(), getTable(),
                idxName, type, collation, names, projects, condition, searchBounds, requiredColumns);
    }

    @Override
    protected IgniteIndexScan copy(
            @Nullable List<RexNode> newProjects,
            @Nullable RexNode newCondition,
            @Nullable List<SearchBounds> newSearchBounds
    ) {
        return new IgniteIndexScan(sourceId, getCluster(), getTraitSet(), getTable(),
                idxName, type, collation, names, newProjects, newCondition, newSearchBounds, requiredColumns);
    }

    /** {@inheritDoc} */
    @Override
    public String getRelTypeName() {
        return REL_TYPE_NAME;
    }

    @Override
    public IgniteRelWriter explain(IgniteRelWriter writer) {
        explainAttributes(writer);

        if (type == Type.SORTED) {
            writer.addCollation(collation, getRowType());
        }

        return writer;
    }
}
