/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.rest;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.jupiter.api.Assertions.assertDoesNotThrow;
import static org.junit.jupiter.api.Assertions.assertThrows;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.micronaut.test.extensions.junit5.annotation.MicronautTest;
import jakarta.inject.Inject;
import java.util.List;
import org.apache.ignite.internal.ClusterPerTestIntegrationTest;
import org.apache.ignite.internal.cli.core.rest.ApiClientFactory;
import org.apache.ignite.rest.client.api.ClusterManagementApi;
import org.apache.ignite.rest.client.invoker.ApiClient;
import org.apache.ignite.rest.client.invoker.ApiException;
import org.apache.ignite.rest.client.model.InitCommand;
import org.apache.ignite.rest.client.model.Problem;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;

/**
 * Test for autogenerated ignite rest client.
 */
@MicronautTest(rebuildContext = true)
public class ItGeneratedRestClientInitTest extends ClusterPerTestIntegrationTest {

    /** Start rest server port. */
    private static final int BASE_REST_PORT = 10300;

    private ClusterManagementApi clusterManagementApi;

    private ObjectMapper objectMapper;

    private String firstNodeName;

    @Inject
    private ApiClientFactory clientFactory;

    @Override
    protected int initialNodes() {
        return 0;
    }

    @BeforeEach
    void setUp(TestInfo testInfo) {
        cluster.startEmbeddedNode(0);
        cluster.startEmbeddedNode(1);
        cluster.startEmbeddedNode(2);

        firstNodeName = cluster.nodeName(0);

        ApiClient apiClient = clientFactory.getClient("http://localhost:" + BASE_REST_PORT);

        clusterManagementApi = new ClusterManagementApi(apiClient);

        objectMapper = new ObjectMapper();
    }

    @Test
    void initCluster() {
        assertDoesNotThrow(() -> clusterManagementApi.init(
                new InitCommand()
                        .clusterName("cluster")
                        .metaStorageNodes(List.of(firstNodeName))
                        .cmgNodes(List.of())
        ));
    }

    @Test
    void initClusterNoSuchNode() throws JsonProcessingException {
        ApiException thrown = assertThrows(
                ApiException.class,
                () -> clusterManagementApi.init(
                        new InitCommand()
                                .clusterName("cluster")
                                .metaStorageNodes(List.of("no-such-node"))
                                .cmgNodes(List.of()))
        );

        assertThat(thrown.getCode(), equalTo(400));

        Problem problem = objectMapper.readValue(thrown.getResponseBody(), Problem.class);
        assertThat(problem.getStatus(), equalTo(400));
        assertThat(problem.getDetail(), containsString("Node \"no-such-node\" is not present in the physical topology"));
    }

    @Test
    void initClusterNoMsCmg() {
        assertDoesNotThrow(() -> clusterManagementApi.init(
                new InitCommand()
                        .clusterName("cluster")
        ));
    }
}
