"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const Observable_1 = require("rxjs/Observable");
const Subject_1 = require("rxjs/Subject");
class Logger extends Observable_1.Observable {
    constructor(name, parent = null) {
        super();
        this.name = name;
        this.parent = parent;
        this._subject = new Subject_1.Subject();
        const path = [];
        let p = parent;
        while (p) {
            path.push(p.name);
            p = p.parent;
        }
        this._metadata = { name, path };
        this._observable = this._subject.asObservable();
        if (this.parent) {
            // When the parent completes, complete us as well.
            this.parent._subject.subscribe(undefined, undefined, () => this.complete());
        }
    }
    get _observable() { return this._obs; }
    set _observable(v) {
        if (this._subscription) {
            this._subscription.unsubscribe();
        }
        this._obs = v;
        if (this.parent) {
            this._subscription = this.subscribe((value) => {
                if (this.parent) {
                    this.parent._subject.next(value);
                }
            }, (error) => {
                if (this.parent) {
                    this.parent._subject.error(error);
                }
            }, () => {
                if (this._subscription) {
                    this._subscription.unsubscribe();
                }
                this._subscription = null;
            });
        }
    }
    asApi() {
        return {
            createChild: (name) => this.createChild(name),
            log: (level, message, metadata) => {
                return this.log(level, message, metadata);
            },
            debug: (message, metadata) => this.debug(message, metadata),
            info: (message, metadata) => this.info(message, metadata),
            warn: (message, metadata) => this.warn(message, metadata),
            error: (message, metadata) => this.error(message, metadata),
            fatal: (message, metadata) => this.fatal(message, metadata),
        };
    }
    createChild(name) {
        return new Logger(name, this);
    }
    complete() {
        this._subject.complete();
    }
    log(level, message, metadata = {}) {
        const entry = Object.assign({}, this._metadata, metadata, {
            level, message, timestamp: +Date.now(),
        });
        this._subject.next(entry);
    }
    debug(message, metadata = {}) {
        return this.log('debug', message, metadata);
    }
    info(message, metadata = {}) {
        return this.log('info', message, metadata);
    }
    warn(message, metadata = {}) {
        return this.log('warn', message, metadata);
    }
    error(message, metadata = {}) {
        return this.log('error', message, metadata);
    }
    fatal(message, metadata = {}) {
        return this.log('fatal', message, metadata);
    }
    toString() {
        return `<Logger(${this.name})>`;
    }
    lift(operator) {
        return this._observable.lift(operator);
    }
    subscribe(_observerOrNext, _error, _complete) {
        return this._observable.subscribe.apply(this._observable, arguments);
    }
    forEach(next, PromiseCtor) {
        return this._observable.forEach(next, PromiseCtor);
    }
}
exports.Logger = Logger;
//# sourceMappingURL=data:application/json;base64,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