## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.path = "figures/sobol-process-",
  fig.width = 7,
  fig.height = 5,
  dpi = 150,
  message = FALSE,
  warning = FALSE,
  eval=FALSE
)

LOCAL <- identical(Sys.getenv("LOCAL"), "TRUE")

library(sensitivity)
library(Sobol4R)
set.seed(4669)

## ----process-demo, cache=TRUE, eval=LOCAL-------------------------------------
Sobol4R:::one_unit(
  lambda1 = 1 / 60,
  lambda2 = 1 / 1012,
  lambda3 = 1 / 72,
  p1      = 0.18,
  p2      = 0.5
)

## ----process-indiv-demo, cache=TRUE, eval=LOCAL-------------------------------
process_fun_indiv(
  X_indiv = c(
    lambda1 = 1 / 60,
    lambda2 = 1 / 1012,
    lambda3 = 1 / 72,
    p1      = 0.18,
    p2      = 0.5
  ),
  M = 50
)

## ----process-design, cache=TRUE, eval=LOCAL-----------------------------------
n <- 200

draw_params <- function(n) {
  data.frame(t(replicate(
    n,
    c(
      1 / runif(1, min = 20,  max = 100),
      1 / runif(1, min = 24,  max = 2000),
      1 / runif(1, min = 24,  max = 120),
      runif(1,  min = 0.05, max = 0.3),
      runif(1,  min = 0.3,  max = 0.7)
    )
  )))
}

X1 <- draw_params(n)
X2 <- draw_params(n)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
gensol_proc <- sobol4r_design(
  X1    = X1,
  X2    = X2,
  order = 2,
  nboot = 100
)
gensol_proc_s2007 <- sensitivity::sobol2007(
  model=NULL,
  X1    = X1,
  X2    = X2,
  nboot = 100
)

## ----process-sobol-single, cache=TRUE, eval=LOCAL-----------------------------
Yproc_1 <- process_fun_row_wise(gensol_proc$X, M = 50)

## ----process-sobol-single2, cache=TRUE, eval=LOCAL----------------------------
Yproc_s2007 <- process_fun_row_wise(gensol_proc_s2007$X, M = 50)

## ----process-sobol-tell, cache=TRUE, eval=LOCAL-------------------------------
xproc_1 <- tell(gensol_proc, Yproc_1)
xproc_s2007 <- tell(gensol_proc_s2007, Yproc_s2007)

## ----process-sobol-single-plot, cache=TRUE, eval=LOCAL------------------------
print(xproc_1)
Sobol4R::autoplot(xproc_1, ncol = 1)

## ----process-sobol-single-plot2, cache=TRUE, eval=LOCAL-----------------------
print(xproc_s2007)
Sobol4R::autoplot(xproc_s2007)

## ----process-sobol-qoi, eval=LOCAL, cache=TRUE--------------------------------
Yproc_2 <- process_fun_mean_to_M(gensol_proc$X, M = 50)

## ----process-sobol-qoi2, cache=TRUE, eval=LOCAL-------------------------------
xproc_2 <- tell(gensol_proc, Yproc_2)

## ----process-sobol-qoi-plot, cache=TRUE, eval=LOCAL---------------------------
print(xproc_2)
Sobol4R::autoplot(xproc_2, ncol = 1)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
res_sobol_mean <- sobol4r_qoi_indices(
  model = process_fun_row_wise,
  X1      = X1,
  X2      = X2,
  qoi_fun = base::mean,
  nrep    = 1000, 
  order   = 2,
  nboot   = 20,
  M       = 50,
  type  = "sobol"
)
print(res_sobol_mean)
Sobol4R::autoplot(res_sobol_mean, ncol = 1)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
res_sobol2007_mean <- sobol4r_qoi_indices(
  model = process_fun_row_wise,
  X1      = X1,
  X2      = X2,
  qoi_fun = base::mean,
  nrep    = 1000, 
  nboot   = 20,
  M       = 50,
  type  = "sobol2007"
)
print(res_sobol2007_mean)
Sobol4R::autoplot(res_sobol2007_mean)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
res_sobol_median <- sobol4r_qoi_indices(
  model = process_fun_row_wise,
  X1      = X1,
  X2      = X2,
  qoi_fun = stats::median,
  nrep    = 1000, 
  order   = 2,
  nboot   = 20,
  M       = 50,
  type    = "sobol"
)
print(res_sobol_median)
Sobol4R::autoplot(res_sobol_median, ncol = 1)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
res_sobol2007_median <- sobol4r_qoi_indices(
  model = process_fun_row_wise,
  X1      = X1,
  X2      = X2,
  qoi_fun = stats::median,
  nrep    = 1000, 
  nboot   = 20,
  M       = 50,
  type    = "sobol2007"
)
print(res_sobol2007_median)
Sobol4R::autoplot(res_sobol2007_median)

